<?php

/**
 * Elementor Single Widget
 * @package goyto Tools
 * @since 1.0.0
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly.

class Goyto_Header extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve Elementor widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'goyto-header';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Elementor widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Goyto Header', 'goyto-plugin' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Elementor widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'goyto-custom-icon';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Elementor widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'goyto_widgets' ];
	}


	protected function register_controls() {

		$this->start_controls_section(
			'print_header_layout',
			[
				'label' => esc_html__( 'Header Layout', 'goyto-plugin' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);
		$this->add_control(
			'style',
			[
				'label' => esc_html__( 'Style', 'goyto-plugin' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => '1',
				'options' => [
					'1'  => esc_html__( 'Header 1', 'goyto-plugin' ),
					'2'  => esc_html__( 'Header 2', 'goyto-plugin' ),
					'3'  => esc_html__( 'Header 3', 'goyto-plugin' ),
					'4'  => esc_html__( 'Header 4', 'goyto-plugin' ),
					'5'  => esc_html__( 'Header 5', 'goyto-plugin' ),
					'6'  => esc_html__( 'Header 6', 'goyto-plugin' ),
					'7'  => esc_html__( 'Header 7', 'goyto-plugin' ),
					'8'  => esc_html__( 'Header 8', 'goyto-plugin' )
				]
			]
		);
		$this->end_controls_section();
		$this->start_controls_section(
			'menu_btn_select',
			[
				'label' => esc_html__( 'Header Button Option', 'goyto-plugin' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);
		$this->add_control(
			'btn_icon_enable',
			[
				'label' => esc_html__( 'Btn Icon Enable', 'goyto-plugin' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'YES', 'goyto-plugin' ),
				'label_off' => esc_html__( 'No', 'goyto-plugin' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);
		
		$this->add_control(
			'btn_icon', [
				'label' => esc_html__( 'Button Icon', 'goyto-plugin' ),
				'type' => Controls_Manager::ICONS,
                'label_block' => true,
				'condition' => [
					'btn_icon_enable' => ['yes'],
				],
			]
		);
		$this->add_control(
			'btn_label', [
				'label' => esc_html__( 'Header Button Label', 'goyto-plugin' ),
				'default' => esc_html__( 'Let’s Talk', 'goyto-plugin' ),
				'type' => Controls_Manager::TEXT,
                'label_block' => true,
				'condition' => [
					'style' => ['1', '2', '6'],
				],
			]
		);
		$this->add_control(
			'btn_link', [
				'label' => esc_html__( 'Header Button Link', 'goyto-plugin' ),
				'type' => Controls_Manager::URL,
                'label_block' => true,
				'condition' => [
					'style' => ['1', '2', '6'],
				],
			]
		);
		
		$this->end_controls_section();

        $this->start_controls_section(
			'logoop',
			[
				'label' => esc_html__( 'Logo Option', 'goyto-plugin' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'rzlogo', [
				'label' => esc_html__( 'Logo', 'goyto-plugin' ),
				'type' => Controls_Manager::MEDIA,
                'label_block' => true,
				'default'     => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
			]
		);
		
		$this->add_responsive_control(
			'logo_max_width',
			[
				'label' => esc_html__( 'Max Width', 'goyto-plugin' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'em', 'rem', 'custom' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 5000,
					]
				],

				'selectors' => [
					'{{WRAPPER}} .logo_site-size' => 'max-width: {{SIZE}}{{UNIT}};',
				],
			]
		);
        
		$this->end_controls_section();

        $this->start_controls_section(
			'menu_select',
			[
				'label' => esc_html__( 'Menu Option', 'goyto-plugin' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);
		$this->add_control(
			'choose-menu',
			[
				'label' => esc_html__( 'Choose menu', 'goyto-plugin' ),
				'type' => \Elementor\Controls_Manager::SELECT2,
				'options' => goyto_menu_selector(),
				'multiple' => false
			]
		);

		$this->end_controls_section();

        $this->start_controls_section(
			'mobile_menu_opt',
			[
				'label' => esc_html__( 'Mobile Option', 'goyto-plugin' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'mrzlogo', [
				'label' => esc_html__( 'Mobile Logo', 'goyto-plugin' ),
				'type' => Controls_Manager::MEDIA,
                'label_block' => true,
				'default'     => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
			]
		);
		$this->add_responsive_control(
			'logo_max_m_width',
			[
				'label' => esc_html__( 'Max Width', 'goyto-plugin' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'em', 'rem', 'custom' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 5000,
					]
				],

				'selectors' => [
					'{{WRAPPER}} .logo img' => 'max-width: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$repeater = new \Elementor\Repeater();

        $repeater->add_control(
			'icons', [
				'label' => esc_html__( 'Icon', 'goyto-plugin' ),
				'type' => Controls_Manager::ICONS,
                'label_block' => true,
			]
		);
        
        $repeater->add_control(
			'link', [
				'label' => esc_html__( 'Link', 'goyto-plugin' ),
				'type' => Controls_Manager::URL,
                'label_block' => true,
			]
		);

        $this->add_control(
			'socials',
			[
				'label' => esc_html__( 'Add Social Item', 'goyto-plugin' ),
				'type' => \Elementor\Controls_Manager::REPEATER,
				'default' => [
					[
						'icons' => [
							'value'   => 'fab fa-facebook-f',
                    		'library' => 'solid',
						],
						'name' => esc_html__( 'Facebook', 'goyto-plugin' ),
						'link' => esc_html__( 'https://facebook.com', 'goyto-plugin' ),
					],
					[
						'icons' => [
							'value'   => 'fab fa-instagram',
                    		'library' => 'solid',
						],
						'name' => esc_html__( 'Instagram', 'goyto-plugin' ),
						'link' => esc_html__( 'https://instagram.com', 'goyto-plugin' ),
					],
					[
						'icons' => [
							'value'   => 'fab fa-youtube',
                    		'library' => 'solid',
						],
						'name' => esc_html__( 'Youtube', 'goyto-plugin' ),
						'link' => esc_html__( 'https://youtube.com', 'goyto-plugin' ),
					],
					[
						'icons' => [
							'value'   => 'fab fa-linkedin',
                    		'library' => 'solid',
						],
						'name' => esc_html__( 'Linkedin', 'goyto-plugin' ),
						'link' => esc_html__( 'https://linkedin.com', 'goyto-plugin' ),
					]
				],
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ name }}}',
			]
		);
		$this->add_control(
			'gallery',
			[
				'label' => esc_html__( 'Add Images', 'goyto-plugin' ),
				'type' => \Elementor\Controls_Manager::GALLERY,
				'show_label' => false,
				'default' => [],
			]
		);
		$this->end_controls_section();

		$this->start_controls_section(
			'menu_bar_style',
			[
				'label' => esc_html__( 'Menu  Style', 'goyto-plugin' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'--menu--style-info',
			[
				'label' => esc_html__( 'Menu Style', 'goyto-plugin' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);
		
        $this->add_control(
			'menu_color',
			[
				'label' => esc_html__( 'Menu Color', 'goyto-plugin' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .main-navigation li a' => 'color: {{VALUE}}'
				],
			]
		);
        $this->add_control(
			'menu_color-hover',
			[
				'label' => esc_html__( 'Menu Hover Color', 'goyto-plugin' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gt-header-section.header_style_one .main-navigation li:hover a' => 'color: {{VALUE}}'
				],
			]
		);
		
        $this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'menu_fb_typography',
				'selector' => '
					{{WRAPPER}} .main-navigation li a
				',
			]
		);
		$this->add_control(
			'menu_dropdown_style',
			[
				'label' => esc_html__( 'Menu Dropdown Style', 'goyto-plugin' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);
		
        $this->add_control(
			'mdb_color',
			[
				'label' => esc_html__( 'Dropdown BG Color', 'goyto-plugin' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gt-header-section .dropdown-menu' => 'background: {{VALUE}}'
				],
			]
		);
        $this->add_control(
			'mdb_menu_color',
			[
				'label' => esc_html__( 'Dropdown Menu Color', 'goyto-plugin' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gt-header-section .dropdown-menu a' => 'color: {{VALUE}}'
				],
			]
		);
		
        $this->end_controls_section();
		$this->start_controls_section(
			'mobile_menu_bg',
			[
				'label' => esc_html__( 'Mobile Menu Style', 'goyto-plugin' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'mb_txt_color',
			[
				'label' => esc_html__( 'Mobile Menu Text Color', 'goyto-plugin' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mobile_menu_content .mobile-main-navigation .navbar-nav li a' => 'color: {{VALUE}}',
					'{{WRAPPER}} .mobile_menu .mobile_menu_content .mobile-main-navigation .navbar-nav .dropdown-menu li a' => 'color: {{VALUE}} !important'
				],
			]
		);
		$this->add_control(
			'mb_bg_color',
			[
				'label' => esc_html__( 'Mobile Menu BG Color', 'goyto-plugin' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mobile_menu_content' => 'background-color: {{VALUE}}'
				],
			]
		);
        $this->end_controls_section();
		$this->start_controls_section(
			'--btn--style-wrapper',
			[
				'label' => esc_html__( 'Button Style', 'goyto-plugin' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
        $this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'm_b_typography',
				'selector' => '
					{{WRAPPER}} .gt-header-section.header_style_one .header-cta-btn,
				
				',
			]
		);
        $this->add_control(
			'padding',
			[
				'label' => esc_html__( 'Padding', 'goyto-plugin' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .gt-header-section.header_style_one .header-cta-btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]
		);
        $this->add_control(
			'b_round',
			[
				'label' => esc_html__( 'Border Radius', 'goyto-plugin' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .gt-header-section.header_style_one .header-cta-btn' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]
		);
        $this->add_control(
			'btn_text',
			[
				'label' => esc_html__( 'Text Color', 'goyto-plugin' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gt-header-section.header_style_one .header-cta-btn' => 'color: {{VALUE}}',
				],
			]
		);

        $this->add_group_control(
			\Elementor\Group_Control_Background::get_type(),
			[
				'name' => 'button_bg_color',
				'types' => [ 'gradient' ],
				'exclude' => [ 'image' ],
				'selector' => '{{WRAPPER}} .gt-header-section.header_style_one .header-cta-btn',
                'fields_options' => [
                    'background' => [
                        'label' => esc_html__( 'Button BG Color ', 'goyto-plugin' ),
                        'description' => esc_html__( 'Choose background type and style.', 'goyto-plugin' ),
                        'separator' => 'before',
                    ]
                ]
			]
		);
        
        $this->end_controls_section();



	}


	protected function render() {
		$settings = $this->get_settings_for_display();
		extract( $settings );
		if ( ! empty( $settings['btn_link']['url'] ) ) {
			$this->add_link_attributes( 'btn_link', $settings['btn_link'] );
		}

		require __DIR__ . '/view/header/header-' . $style . '.php';
    }

	/**
	 * Search Body Display
	 *
	 * @return void
	 */
	protected function ___search_body(){
		$settings = $this->get_settings_for_display();
		?>
		<div class="search-body">
			<div class="search-form">
				<form action="<?php print esc_url(home_url('/')); ?>" class="search-form-area">
					
					<input class="search-input" required type="search" name="s" aria-label="search" placeholder="<?php esc_attr_e( 'Search Keywords..', 'goyto-plugin' );?>" value="<?php print esc_attr( get_search_query() )?>">
					<button type="submit" class="search-btn1">
						<i class="fas fa-search"></i>
					</button>	
				</form>
				<div class="outer-close text-center search-btn">
					<i class="far fa-times"></i>
				</div>
			</div>
		</div>
<?php 
	}



	/**
	 * Mobile Menu Display
	 *
	 * @return void
	 */
	protected function __mobile_menu(){ 
		$settings = $this->get_settings_for_display();
	?>
	<div class="mobile_menu lenis lenis-smooth position-relative">
		<div class="mobile_menu_wrap">
			<div class="mobile_menu_overlay open_mobile_menu"></div>
			<div class="mobile_menu_content">
				<div class="mobile_menu_close open_mobile_menu">
					<i class="fal fa-times"></i>
				</div>
				<div class="m-brand-logo">
					<a href="<?php echo esc_url(home_url());?>">
						<img class="logo_site-size" src="<?php echo esc_url($settings['mrzlogo']['url']);?>" alt="<?php if(!empty($settings['mrzlogo']['alt'])){ echo esc_attr($settings['mrzlogo']['alt']);}?>">
					</a>
				</div>
				<div class="mobile-search-bar position-relative">
					<form action="<?php echo esc_url(home_url( '/' ));?>">
						<input type="text" name="search" placeholder="<?php esc_attr_e( 'Search Keywords..', 'goyto-plugin' );?>" value="<?php echo get_search_query();?>">
						<button><i class="fal fa-search"></i></button>
					</form>
				</div>
				<nav class="mobile-main-navigation  clearfix ul-li">
					<?php
						echo str_replace(['menu-item-has-children', 'sub-menu'], ['dropdown', 'dropdown-menu clearfix'], wp_nav_menu( array(
							'echo'           => false,
							'menu' => !empty($settings['choose-menu']) ? $settings['choose-menu'] : 'menu-1',
							'menu_id'        =>'m-main-nav',
							'menu_class'        =>'navbar-nav text-capitalize clearfix',
							'container'=>false,
							'fallback_cb'    => 'Navwalker_Class::fallback',
							'walker'         => class_exists( 'Rs_Mega_Menu_Walker' ) ? new \Rs_Mega_Menu_Walker : '',
						)) );
					?>
				</nav>
				<?php if(!empty($settings['hsocials'])):?>
				<div class="ptx-mobile-header-social text-center">
					<?php foreach($settings['socials'] as $item):?>
						<a href="<?php echo esc_url($item['link']['url'])?>"> <?php \Elementor\Icons_Manager::render_icon( $item['icons'], [ 'aria-hidden' => 'true' ] ); ?></a></a>
					<?php endforeach;?>
				</div>
				<?php endif;?>
			</div>
		</div>
		<!-- /Mobile-Menu -->
	</div>
	<?php }


}


Plugin::instance()->widgets_manager->register( new Goyto_Header() );